<?php
declare(strict_types=1);

require __DIR__ . '/../lib/db.php';
require __DIR__ . '/../lib/auth.php';
require __DIR__ . '/../lib/utils.php';
require __DIR__ . '/../lib/rate_limit.php';

auth_boot();
require_login();

$cfg = require __DIR__ . '/../config/config.php';
rate_limit_or_die('chat', (int)$cfg['app']['rate_limit_per_hour']);

$in = json_input();
$msg = trim((string)($in['message'] ?? ''));
if ($msg === '') json_out(['ok'=>false,'error'=>'Message vide.'], 400);

try {
  $command = ($cfg['mode'] === 'ai') ? ai_to_command($msg, $cfg) : rules_to_command($msg);
  $reply = apply_command($command);
  json_out(['ok'=>true,'reply'=>$reply, 'command'=>$command]);
} catch (Throwable $e) {
  json_out(['ok'=>false,'error'=>$e->getMessage()], 400);
}

function ai_to_command(string $msg, array $cfg): array {
  $key = 'ai_calls_' . date('Ymd');
  if (!isset($_SESSION[$key])) $_SESSION[$key] = 0;
  $_SESSION[$key]++;
  if ($_SESSION[$key] > (int)$cfg['ai']['max_daily_calls']) {
    return rules_to_command($msg);
  }

  $system = <<<SYS
Tu es un assistant de gestion pour une librairie en Côte d'Ivoire.
Ta mission: transformer le message utilisateur en une commande JSON STRICTE.
Tu ne dois renvoyer QUE du JSON valide, sans texte supplémentaire.

Schéma attendu:
{
 "intent": "sale|expense|stock_in|stock_check|low_stock|summary",
 "items": [{"name": "...", "qty": 1, "unit_price": 0}],
 "amount": 0,
 "payment_method": "cash|mobile_money|card|other",
 "note": ""
}

Règles:
- Si c'est une vente: intent="sale", items requis (name, qty, unit_price si mentionné).
- Si c'est une dépense: intent="expense", amount requis + note courte (ex: transport).
- Si c'est un ajout de stock: intent="stock_in", items requis (name, qty).
- "Montre rupture" => intent="low_stock"
- "Résumé du jour" => intent="summary"
- "Stock de X ?" => intent="stock_check" + items[0].name = X
- Devise: FCFA. Si prix non fourni, mets unit_price=0.
SYS;

  $payload = [
    'model' => $cfg['ai']['model'],
    'messages' => [
      ['role'=>'system','content'=>$system],
      ['role'=>'user','content'=>"Message: " . $msg]
    ],
    'temperature' => 0.1
  ];

  $resp = http_post_json("https://api.openai.com/v1/chat/completions", $payload, [
    "Authorization: Bearer {$cfg['ai']['api_key']}",
  ]);

  if (!isset($resp['choices'][0]['message']['content'])) {
    return rules_to_command($msg);
  }

  $content = trim((string)$resp['choices'][0]['message']['content']);
  $decoded = json_decode($content, true);
  if (!is_array($decoded) || empty($decoded['intent'])) {
    return rules_to_command($msg);
  }
  return $decoded;
}

function http_post_json(string $url, array $payload, array $headers = []): array {
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => array_merge(['Content-Type: application/json'], $headers),
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_TIMEOUT => 20
  ]);
  $out = curl_exec($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  if ($out === false) throw new RuntimeException('Erreur réseau API.');
  curl_close($ch);

  $json = json_decode($out, true);
  if ($code >= 400) throw new RuntimeException('AI indisponible (ou clé invalide).');
  return is_array($json) ? $json : [];
}

function rules_to_command(string $msg): array {
  $m = mb_strtolower($msg);

  if (str_contains($m, 'résumé') || str_contains($m, 'resume')) return ['intent'=>'summary'];
  if (str_contains($m, 'rupture') || str_contains($m, 'en rupture') || str_contains($m, 'stock faible')) return ['intent'=>'low_stock'];

  if (preg_match('/(d[ée]pense)\s+(\d+)/u', $m, $mm)) {
    $amount = (float)$mm[2];
    $note = trim(preg_replace('/(d[ée]pense)\s+\d+/u', '', $msg));
    return ['intent'=>'expense','amount'=>$amount,'payment_method'=>'cash','note'=>$note ?: 'dépense'];
  }

  if (preg_match('/(ajoute|stock)\s+(\d+)\s+(.+)/u', $m, $mm)) {
    return ['intent'=>'stock_in','items'=>[['name'=>trim($mm[3]),'qty'=>(int)$mm[2],'unit_price'=>0]], 'payment_method'=>'cash','note'=>''];
  }

  if (preg_match('/(vendu|vente)\s+(\d+)\s+(.+?)(?:\s+[àa]\s+(\d+))?$/u', $m, $mm)) {
    $name = trim($mm[3]);
    $price = isset($mm[4]) ? (float)$mm[4] : 0;
    return ['intent'=>'sale','items'=>[['name'=>$name,'qty'=>(int)$mm[2],'unit_price'=>$price]], 'payment_method'=>'cash','note'=>''];
  }

  if (str_starts_with($m, 'stock')) {
    $name = trim(preg_replace('/^stock\s*(de)?/u', '', $msg));
    return ['intent'=>'stock_check','items'=>[['name'=>$name,'qty'=>1,'unit_price'=>0]]];
  }

  return ['intent'=>'unknown','note'=>$msg];
}

function apply_command(array $cmd): string {
  $intent = $cmd['intent'] ?? 'unknown';

  return match ($intent) {
    'sale' => apply_sale($cmd),
    'expense' => apply_expense($cmd),
    'stock_in' => apply_stock_in($cmd),
    'low_stock' => show_low_stock(),
    'stock_check' => show_stock($cmd),
    'summary' => show_summary(),
    default => "Je n'ai pas compris. Exemples: “Vendu 3 cahiers 200 pages à 500 FCFA”, “Dépense 2000 transport”, “Ajoute 20 cahiers 200 pages”, “Fais le résumé du jour”."
  };
}

function find_or_create_product(string $rawName): array {
  $name = normalize_name($rawName);
  if ($name === '') throw new RuntimeException('Nom produit manquant.');

  $stmt = db()->prepare("SELECT * FROM products WHERE name = ? LIMIT 1");
  $stmt->execute([$name]);
  $p = $stmt->fetch();
  if ($p) return $p;

  $ins = db()->prepare("INSERT INTO products (name, category, sell_price, stock_qty, reorder_level) VALUES (?, 'Livre', 0, 0, 0)");
  $ins->execute([$name]);

  $stmt->execute([$name]);
  return $stmt->fetch() ?: ['id'=>db()->lastInsertId(),'name'=>$name,'sell_price'=>0,'stock_qty'=>0,'reorder_level'=>0];
}

function apply_sale(array $cmd): string {
  $items = $cmd['items'] ?? [];
  if (!is_array($items) || count($items) === 0) throw new RuntimeException('Vente: aucun article détecté.');

  $pm = $cmd['payment_method'] ?? 'cash';
  $uid = current_user_id();

  $lines = [];
  $total = 0.0;

  foreach ($items as $it) {
    $p = find_or_create_product((string)($it['name'] ?? ''));
    $qty = (int)($it['qty'] ?? 1);
    $unit = (float)($it['unit_price'] ?? 0);
    if ($qty <= 0) $qty = 1;

    if ($unit <= 0) $unit = (float)$p['sell_price'];

    $amount = $qty * $unit;
    $total += $amount;

    $stmt = db()->prepare("INSERT INTO transactions (type, product_id, quantity, unit_price, amount, payment_method, note, created_by)
                           VALUES ('sale', ?, ?, ?, ?, ?, ?, ?)");
    $stmt->execute([(int)$p['id'], $qty, $unit, $amount, $pm, '', $uid]);

    $upd = db()->prepare("UPDATE products SET stock_qty = stock_qty - ? WHERE id = ?");
    $upd->execute([$qty, (int)$p['id']]);

    $st = db()->prepare("SELECT stock_qty, reorder_level FROM products WHERE id=?");
    $st->execute([(int)$p['id']]);
    $row = $st->fetch() ?: ['stock_qty'=>0,'reorder_level'=>0];

    $warn = ((int)$row['stock_qty'] <= (int)$row['reorder_level'] && (int)$row['reorder_level'] > 0)
      ? " ⚠️ Stock faible ({$row['stock_qty']})." : "";

    $lines[] = "✅ Vente enregistrée: {$qty} × {$p['name']} = " . (int)$amount . " FCFA. Stock: {$row['stock_qty']}." . $warn;
  }

  $lines[] = "Total vente: " . (int)$total . " FCFA.";
  return implode("\n", $lines);
}

function apply_expense(array $cmd): string {
  $amount = (float)($cmd['amount'] ?? 0);
  if ($amount <= 0) throw new RuntimeException('Dépense: montant manquant.');
  $note = trim((string)($cmd['note'] ?? 'dépense'));
  $pm = $cmd['payment_method'] ?? 'cash';
  $uid = current_user_id();

  $stmt = db()->prepare("INSERT INTO transactions (type, amount, payment_method, note, created_by)
                         VALUES ('expense', ?, ?, ?, ?)");
  $stmt->execute([$amount, $pm, $note, $uid]);

  return "✅ Dépense enregistrée: " . (int)$amount . " FCFA — " . ($note ?: 'dépense') . ".";
}

function apply_stock_in(array $cmd): string {
  $items = $cmd['items'] ?? [];
  if (!is_array($items) || count($items) === 0) throw new RuntimeException('Stock: aucun article détecté.');

  $uid = current_user_id();
  $lines = [];

  foreach ($items as $it) {
    $p = find_or_create_product((string)($it['name'] ?? ''));
    $qty = (int)($it['qty'] ?? 0);
    if ($qty <= 0) throw new RuntimeException('Stock: quantité invalide.');

    $stmt = db()->prepare("INSERT INTO transactions (type, product_id, quantity, amount, payment_method, note, created_by)
                           VALUES ('stock_in', ?, ?, 0, 'cash', '', ?)");
    $stmt->execute([(int)$p['id'], $qty, $uid]);

    $upd = db()->prepare("UPDATE products SET stock_qty = stock_qty + ? WHERE id = ?");
    $upd->execute([$qty, (int)$p['id']]);

    $st = db()->prepare("SELECT stock_qty FROM products WHERE id=?");
    $st->execute([(int)$p['id']]);
    $row = $st->fetch() ?: ['stock_qty'=>0];

    $lines[] = "✅ Stock ajouté: +{$qty} {$p['name']}. Stock: {$row['stock_qty']}.";
  }

  return implode("\n", $lines);
}

function show_low_stock(): string {
  $rows = db()->query("SELECT name, stock_qty, reorder_level
                       FROM products
                       WHERE active=1 AND reorder_level > 0 AND stock_qty <= reorder_level
                       ORDER BY stock_qty ASC, name ASC
                       LIMIT 20")->fetchAll();

  if (!$rows) return "✅ Aucun article en alerte (stock faible).";

  $out = "⚠️ Articles en alerte (stock ≤ seuil):\n";
  foreach ($rows as $r) {
    $out .= "- {$r['name']}: {$r['stock_qty']} (seuil {$r['reorder_level']})\n";
  }
  return trim($out);
}

function show_stock(array $cmd): string {
  $items = $cmd['items'] ?? [];
  $name = (string)($items[0]['name'] ?? '');
  $name = normalize_name($name);
  if ($name === '') return "Indique le produit. Exemple: “Stock de cahier 200 pages”";

  $stmt = db()->prepare("SELECT name, stock_qty, sell_price, reorder_level FROM products WHERE name=? LIMIT 1");
  $stmt->execute([$name]);
  $p = $stmt->fetch();
  if (!$p) return "Produit introuvable. Ajoute-le via Admin, ou dis: “Ajoute 10 {$name}” (il sera créé).";

  $warn = ((int)$p['reorder_level'] > 0 && (int)$p['stock_qty'] <= (int)$p['reorder_level']) ? " ⚠️ Stock faible." : "";
  return "📦 {$p['name']} — Stock: {$p['stock_qty']}, Prix: " . (int)$p['sell_price'] . " FCFA." . $warn;
}

function show_summary(): string {
  $date = today_date();

  $sales = db()->prepare("SELECT COALESCE(SUM(amount),0) AS s FROM transactions WHERE type='sale' AND DATE(created_at)=?");
  $sales->execute([$date]);
  $totalSales = (float)($sales->fetch()['s'] ?? 0);

  $exp = db()->prepare("SELECT COALESCE(SUM(amount),0) AS e FROM transactions WHERE type='expense' AND DATE(created_at)=?");
  $exp->execute([$date]);
  $totalExp = (float)($exp->fetch()['e'] ?? 0);

  $net = $totalSales - $totalExp;

  $top = db()->prepare("
    SELECT p.name, SUM(t.quantity) AS qty, SUM(t.amount) AS amt
    FROM transactions t
    JOIN products p ON p.id=t.product_id
    WHERE t.type='sale' AND DATE(t.created_at)=?
    GROUP BY p.id
    ORDER BY amt DESC
    LIMIT 5
  ");
  $top->execute([$date]);
  $topRows = $top->fetchAll();

  $out = "📌 Résumé du jour ({$date})\n";
  $out .= "• Chiffre d'affaires: " . (int)$totalSales . " FCFA\n";
  $out .= "• Dépenses: " . (int)$totalExp . " FCFA\n";
  $out .= "• Solde net (cash estimé): " . (int)$net . " FCFA\n";

  if ($topRows) {
    $out .= "• Top ventes:\n";
    foreach ($topRows as $r) {
      $out .= "  - {$r['name']}: {$r['qty']} (".(int)$r['amt']." FCFA)\n";
    }
  }

  return trim($out);
}
